const {version} = require("../../../package.json")
const config = require("../../lib/config")
const logger = require("../../lib/logger")
const log = logger(config.logger)
const {upsertDataSource, getDataSource, deleteDataSource, getEvents} = require("./repo")

module.exports.getV1ConnectorInfo = async () => {
    return {
        status: 200,
        data: {
            kind: "sample-issue-tracker",
            label: "Example Issue Tracker from `tests/sample-external-data-connector`",
            version,
            configSchema: {
                options: [{name: "projectName", description: "The name of the project (will be added as a prefix to issues keys)"}],
            },
        },
    }
}

module.exports.putV1ConnectorDataSourcesById = async ({id, body}) => {
    const {projectName} = body?.config
    if (!projectName)
        return {
            status: 400,
            data: "missing projectName",
        }
    upsertDataSource(id, projectName)
    log.info(`Data Source created: ${id} (${projectName})`)
    return {status: 201}
}

module.exports.deleteV1ConnectorDataSourcesById = async ({id}) => {
    const ds = getDataSource(id)
    if (!ds) return {status: 404}
    deleteDataSource(id)
    log.info(`Data Source deleted: ${id}`)
    return {status: 200}
}

module.exports.getV1ConnectorDataSourcesByIdInfo = async ({id}) => {
    const ds = getDataSource(id)
    if (!ds) return {status: 404}
    return {
        status: 200,
        data: {
            id,
            label: `Example issue project ${ds}`,
            kind: "sample-issue-tracker",
            initialPosition: "0",
            entities: [
                {
                    type: "issue",
                    fields: [
                        {id: "f_key", name: "key", description: "The issue key", fieldType: {type: "Text"}},
                        {id: "f_status", name: "status", description: "Status of the issue", fieldType: {type: "Label", multiple: false}},
                        {id: "f_summary", name: "summary", description: "Title of the issue", fieldType: {type: "Text"}},
                    ],
                    references: [
                        {
                            id: "r_assignee",
                            name: "assignee",
                            description: "Reponsible person for the issue",
                            types: ["user"],
                            multiple: false,
                        },
                    ],
                },
                {
                    type: "user",
                    fields: [
                        {id: "f_id", name: "id", description: "The user id", fieldType: {type: "Text"}},
                        {id: "f_name", name: "name", description: "Display name of the user (first + last)", fieldType: {type: "Text"}},
                        {id: "f_email", name: "email", description: "Email of the user", fieldType: {type: "Text"}},
                    ],
                    references: [],
                },
            ],
        },
    }
}

module.exports.getV1ConnectorDataSourcesByIdStatus = async ({id}) => {
    const ds = getDataSource(id)
    if (!ds) return {status: 404}

    return {
        status: 200,
        data: {
            status: "Ok",
            lastPosition: "0",
        },
    }
}

module.exports.getV1ConnectorDataSourcesByIdEvents = async ({id, afterPosition}) => {
    if (!afterPosition) return {status: 400, data: "missing parameter afterPosition"}
    const pos = parseInt(afterPosition, 10)
    const ds = getDataSource(id)
    if (!ds) return {status: 404}

    const events = (getEvents(id) || []).slice(pos, pos + 10)
    if (events.length) console.log(`returning ${events.length} events for ${id} after position ${afterPosition}`)
    return {
        status: 200,
        data: events,
    }
}
