const express = require("express")
const connector = require("../services/connector")

const router = new express.Router()

/**
 * Gets information about the connector and its capabilities.
 */
router.get("/connector/info", async (req, res, next) => {
    const options = {}

    try {
        const result = await connector.getV1ConnectorInfo(options)
        res.status(result.status || 200).send(result.data)
    } catch (err) {
        return res.status(500).send({
            status: 500,
            error: "Server Error",
        })
    }
})

function datasourceFromPath(req) {
    const regex = /\/data-sources\/([^\/]+)(\/|$)/
    const match = req.path.match(regex)
    return match ? match[1] : null
}

/**
 * Creates or reconfigures a data source. The connector is not
 * required to persist this information, Phonemos will
 * retrigger the put if an operation on the data source return
 * 404. But the connector must remember it with a reasonable
 * lifetime (eg until the next restart or so).
 */
router.put("/connector/data-sources/:id", async (req, res, next) => {
    const options = {
        id: datasourceFromPath(req),
        body: req.body,
    }

    try {
        const result = await connector.putV1ConnectorDataSourcesById(options)
        res.status(result.status).send(result.data)
    } catch (err) {
        return res.status(500).send({
            status: 500,
            error: "Server Error",
        })
    }
})

/**
 * The data source has been deleted in Phonemos, the connector
 * can discard any data associated with it.
 */
router.delete("/connector/data-sources/:id", async (req, res, next) => {
    const options = {
        id: datasourceFromPath(req),
    }

    try {
        const result = await connector.deleteV1ConnectorDataSourcesById(options)
        res.status(result.status || 200).send(result.data)
    } catch (err) {
        next(err)
    }
})

/**
 * Gets meta information about the data source and the items it
 * returns.
 */
router.get("/connector/data-sources/:id/info", async (req, res, next) => {
    const options = {
        id: datasourceFromPath(req),
    }

    try {
        const result = await connector.getV1ConnectorDataSourcesByIdInfo(options)
        res.status(result.status || 200).send(result.data)
    } catch (err) {
        next(err)
    }
})

/**
 * Gets the status of the data source
 */
router.get("/connector/data-sources/:id/status", async (req, res, next) => {
    const options = {
        id: datasourceFromPath(req),
    }

    try {
        const result = await connector.getV1ConnectorDataSourcesByIdStatus(options)
        res.status(result.status || 200).send(result.data)
    } catch (err) {
        next(err)
    }
})

/**
 * Events from the data source
 */
router.get("/connector/data-sources/:id/events", async (req, res, next) => {
    const options = {
        id: datasourceFromPath(req),
        afterPosition: req.query.afterPosition,
    }

    try {
        const result = await connector.getV1ConnectorDataSourcesByIdEvents(options)
        res.status(result.status || 200).send(result.data)
    } catch (err) {
        next(err)
    }
})

module.exports = router
